/**
 * Copyright 2015 Yamato
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package mod.ymt.rndsp;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import net.minecraft.block.Block;
import net.minecraft.block.BlockMobSpawner;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.monster.EntitySpider;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.MobSpawnerBaseLogic;
import net.minecraft.tileentity.TileEntityMobSpawner;
import net.minecraft.world.World;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.world.BlockEvent.BreakEvent;
import net.minecraftforge.event.world.BlockEvent.PlaceEvent;
import cpw.mods.fml.common.Mod;
import cpw.mods.fml.common.Mod.EventHandler;
import cpw.mods.fml.common.event.FMLPreInitializationEvent;
import cpw.mods.fml.common.eventhandler.SubscribeEvent;

@Mod(modid = "mod.ymt.rndsp.RandomSpawner", name = "RandomSpawner", version = "17Av1")
public class RandomSpawner {
	private final Random rand = new Random();
	private final Deque<SpawnerEntry> queue = new LinkedList<SpawnerEntry>();
	private final List<SpawnerEntry> mobs;
	{
		// Animals
		List<SpawnerEntry> mobs = new ArrayList<SpawnerEntry>();
		mobs.add(new SpawnerEntry("Cow"));
		mobs.add(new SpawnerEntry("Pig"));
		mobs.add(new SpawnerEntry("Chicken"));
		mobs.add(new SpawnerEntry("Sheep"));
		mobs.add(new SpawnerEntry("Wolf"));
		mobs.add(new SpawnerEntry("MushroomCow"));
		mobs.add(new SpawnerEntry("SnowMan"));
		mobs.add(new SpawnerEntry("Ozelot"));
		mobs.add(new SpawnerEntry("VillagerGolem", "IronGolem"));
		mobs.add(new SpawnerEntry("Villager"));
		mobs.add(new SpawnerEntry("Bat"));
		mobs.add(new SpawnerEntry("Squid"));
		// mobs.add(new SpawnerEntry("EntityHorse"));	// EntityHorse はスポナー非対応 (null world に対応していない
		
		// Monster
		mobs.add(new SpawnerEntry("Creeper"));
		mobs.add(new SpawnerEntry("Skeleton"));
		mobs.add(new SpawnerEntry("Skeleton", "WitherSkeleton") {
			{
				tag.setByte("SkeletonType", (byte) 1);
				setEquipmentTag(tag, Items.stone_sword);
			}
		});
		mobs.add(new SpawnerEntry("Skeleton", "SpiderJockey") {
			{
				EntitySpider spider = new EntitySpider(null);
				NBTTagCompound riding = new NBTTagCompound();
				spider.writeToNBTOptional(riding);
				tag.setTag("Riding", riding);
				setEquipmentTag(tag, Items.bow);
			}
		});
		mobs.add(new SpawnerEntry("Spider"));
		mobs.add(new SpawnerEntry("Zombie"));
		mobs.add(new SpawnerEntry("Slime"));
		mobs.add(new SpawnerEntry("Ghast"));
		mobs.add(new SpawnerEntry("PigZombie"));
		mobs.add(new SpawnerEntry("Enderman"));
		mobs.add(new SpawnerEntry("CaveSpider"));
		mobs.add(new SpawnerEntry("Silverfish"));
		mobs.add(new SpawnerEntry("Blaze"));
		mobs.add(new SpawnerEntry("LavaSlime"));
		mobs.add(new SpawnerEntry("Witch"));
		mobs.add(new SpawnerEntry("WitherBoss", "Wither") {
			{
				tag.setInteger("Invul", 220);
			}
		});
		
		// Other
		mobs.add(new SpawnerEntry("PrimedTnt", "TNT") {
			{
				tag.setByte("Fuse", (byte) 80);
			}
		});
		mobs.add(new SpawnerEntry("FallingSand", "Sand") {
			{
				tag.setByte("Tile", (byte) Block.getIdFromBlock(Blocks.sand));
				tag.setInteger("TileID", Block.getIdFromBlock(Blocks.sand));
				tag.setByte("Time", (byte) 1);
			}
		});
		mobs.add(new SpawnerEntry("FallingSand", "Gravel") {
			{
				tag.setByte("Tile", (byte) Block.getIdFromBlock(Blocks.gravel));
				tag.setInteger("TileID", Block.getIdFromBlock(Blocks.gravel));
				tag.setByte("Time", (byte) 1);
			}
		});
		mobs.add(new SpawnerEntry("XPOrb") {
			{
				tag.setByte("Value", (byte) 5);
			}
		});
		this.mobs = Collections.unmodifiableList(mobs);
	}
	
	@SubscribeEvent
	public void onBreakBlock(BreakEvent e) {
		if (e.getPlayer() != null && e.block instanceof BlockMobSpawner) {
			if (e.getPlayer().capabilities.isCreativeMode) {
				return;
			}
			// MobSpawner を壊したらブロックが落ちるように
			dropBlockAsItem(e.world, e.x, e.y, e.z, new ItemStack(e.block));
		}
	}
	
	@SubscribeEvent
	public void onPlaceBlock(PlaceEvent e) {
		if (e.player != null && e.block instanceof BlockMobSpawner) {
			// 次のモブ取得
			SpawnerEntry entry = getNextEntityName();
			// モブ情報セット
			TileEntityMobSpawner newSpawner = (TileEntityMobSpawner) e.world.getTileEntity(e.x, e.y, e.z);
			entry.apply(newSpawner.func_145881_a());
			// 標準出力
			System.out.println("SpawnerBlock changed: " + entry.displayName);
		}
	}
	
	@EventHandler
	public void preInit(FMLPreInitializationEvent event) {
		MinecraftForge.EVENT_BUS.register(this);
	}
	
	private SpawnerEntry getNextEntityName() {
		synchronized (this) {
			if (queue.isEmpty()) {
				List<SpawnerEntry> list = new ArrayList<SpawnerEntry>(mobs);
				Collections.shuffle(list, rand);
				queue.addAll(list);
			}
			if (!queue.isEmpty()) {
				return queue.removeFirst();
			}
			return new SpawnerEntry("Pig");
		}
	}
	
	private static void dropBlockAsItem(World world, int x, int y, int z, ItemStack item) {
		if (!world.isRemote && world.getGameRules().getGameRuleBooleanValue("doTileDrops") /* && !world.restoringBlockSnapshots */) {
			float f = 0.7F;
			double rx = (double) (world.rand.nextFloat() * f) + (double) (1.0F - f) * 0.5D;
			double ry = (double) (world.rand.nextFloat() * f) + (double) (1.0F - f) * 0.5D;
			double rz = (double) (world.rand.nextFloat() * f) + (double) (1.0F - f) * 0.5D;
			EntityItem entityitem = new EntityItem(world, x + rx, y + ry, z + rz, item);
			entityitem.delayBeforeCanPickup = 10;
			world.spawnEntityInWorld(entityitem);
		}
	}
	
	private static void setEquipmentTag(NBTTagCompound tag, Item item) {
		NBTTagList taglist = new NBTTagList();
		NBTTagCompound tagc = new NBTTagCompound();
		new ItemStack(item).writeToNBT(tagc);
		taglist.appendTag(tagc);
		tag.setTag("Equipment", taglist);
	}
	
	public static class SpawnerEntry {
		public final String entityName;
		public final String displayName;
		public final NBTTagCompound tag = new NBTTagCompound();
		
		public SpawnerEntry(String name) {
			this(name, name);
		}
		
		public SpawnerEntry(String entryName, String displayName) {
			this.entityName = entryName;
			this.displayName = displayName;
		}
		
		public void apply(MobSpawnerBaseLogic spawner) {
			spawner.setEntityName(entityName);
			spawner.setRandomEntity(tag.hasNoTags() ? null : spawner.new WeightedRandomMinecart(tag, entityName));
			// ほんとは setRandomEntity を直接使わずに potentialEntitySpawns を使うのが良さそうだけど面倒なので
			spawner.spawnDelay = 10 * 20;
		}
	}
}
